<?php
/*
	Plugin Name: Gravity Forms Google Spreadsheet Addon
	Description: Add gravity forms data to Google Spreadsheet when form submits.
	Author: Webholics
	Author URI: https://webholics.org
	Plugin URI: https://webholics.org
	Version: 3.1
*/
define( 'GFSPREADSHEET_OAUTH2_REVOKE_URI', 'https://accounts.google.com/o/oauth2/revoke' );
define( 'GFSPREADSHEET_OAUTH2_TOKEN_URI', 'https://www.googleapis.com/oauth2/v4/token' );
define( 'GFSPREADSHEET_OAUTH2_AUTH_URL', 'https://accounts.google.com/o/oauth2/auth' );
define( 'GFSPREADSHEET_VERSION', '3.1' );
add_action( 'gform_loaded', array( 'GF_Spreadsheet_Feed_AddOn_Bootstrap', 'load' ), 5 );
class GF_Spreadsheet_Feed_AddOn_Bootstrap {
	public static function load() {
		if ( ! method_exists( 'GFForms', 'include_feed_addon_framework' ) ) {
			return;
		}
		require_once 'class-gfspreadsheet-auth.php';
		require_once 'class-gfspreadsheetfeedaddon.php';
		require_once 'inc/class-gfspreadsheetreview.php';
		GFAddOn::register( 'GFSpreadsheetFeedAddOn' );
	}
}
function gfspreadsheet_simple_feed_addon() {
	return GFSpreadsheetFeedAddOn::get_instance();
}

class GF_GoogleSpreadsheet_Api {

	/**
	 *
	 * @var WP_GoogleSpreadsheet
	 * @since 2.0
	 */
	private static $instance;

	/**
	 * Plugin Directory
	 *
	 * @since 2.0
	 * @var string $dir
	 */
	public static $dir = '';



	/**
	 * Plugin URL
	 *
	 * @since 2.0
	 * @var string $url
	 */
	public static $url = '';

	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof GF_GoogleSpreadsheet_Api ) ) {
			self::$instance = new GF_GoogleSpreadsheet_Api();

			self::$dir = plugin_dir_path( __FILE__ );

			self::$url = plugin_dir_url( __FILE__ );

		}
		return self::$instance;
	}

	public function get_url(){
		return self::$url;
	}


	/**
	 * Get List of GoogleSpreadsheets
	 *
	 * @since  1.0
	 * @author Aman Saini
	 * @return Files Array
	 */
	function get_googlespreadsheet_files() {
		$cached_files = get_transient( 'gf_spreadsheet_files' );
		if ( empty( $cached_files ) ) {
			$files    = array();
			$response = $this->make_request( 'drive', 'GET', '/drive/v3/files', array( 'q' => "mimeType%3D'application%2Fvnd.google-apps.spreadsheet'" ) );
			if ( is_object( $response ) ) {
				if ( isset( $response->kind ) && $response->kind == 'drive#fileList' ) {
					foreach ( $response->files as $file ) {
						$files[] = array(
							'label' => $file->name,
							'id'    => $file->id,
						);
					}
					set_transient( 'gf_spreadsheet_files', $files, 60 * 60 * 60 );
				} elseif ( isset( $response->error->errors ) ) {
					echo '<pre>';
					foreach ( $response->error->errors as $err ) {
						echo '<b>' . $err->message . '</b> &nbsp; Please check if you have completed the authorization';
						die;
					}
				}
			}
		} else {
			$files = $cached_files;
		}

		return $files;

	}

	/**
	 * Get List of GoogleSpreadsheet Sheets
	 *
	 * @since  1.0
	 * @author Aman Saini
	 * @return Files Array
	 */
	function get_googlespreadsheet_sheets( $spreadsheet_id ) {

		if ( empty( $spreadsheet_id ) ) {
			return;
		}
		$cached_sheets = get_transient( 'gf_spreadsheet_sheets' );

		if ( empty( $cached_sheets[ $spreadsheet_id ] ) ) {
			$sheets   = array();
			$response = $this->make_request( 'sheets', 'GET', '/' . $spreadsheet_id, array( 'includeGridData' => 'false' ) );

			if ( is_object( $response ) ) {
				if ( isset( $response->sheets ) ) {
					foreach ( $response->sheets as $sheet ) {
						$sheets[] = array(
							'label' => $sheet->properties->title,
							'id'    => $sheet->properties->title,
						);
					}
					$cached_sheets[ $spreadsheet_id ] = $sheets;
					set_transient( 'gf_spreadsheet_sheets', $cached_sheets, 60 * 60 * 60 );

				} elseif ( isset( $response->error->errors ) ) {
					echo '<pre>';
					print_r( $response->error->errors );
					die;
				}
			}
		} else {
			$sheets = $cached_sheets[ $spreadsheet_id ];
		}

		return json_encode( $sheets );

	}


	/**
	 * Get List of GoogleSpreadsheet Headers
	 *
	 * @since  1.0
	 * @author Aman Saini
	 * @return Files Array
	 */
	function get_googlespreadsheet_headers( $spreadsheet_id, $sheet_id ) {

		if ( empty( $spreadsheet_id ) || empty( $sheet_id ) ) {
			return '';
		}
		$sheet_id       = (string) $sheet_id;
		$cached_headers = get_transient( 'gf_spreadsheet_headers' );

		if ( empty( $cached_headers[ $spreadsheet_id ][ $sheet_id ] ) ) {
			$headers  = array();
			$response = $this->make_request( 'sheets', 'GET', '/' . $spreadsheet_id . '/values/' . urlencode( $sheet_id ) );

			if ( is_object( $response ) ) {
				if ( isset( $response->values[0] ) ) {
					foreach ( $response->values[0] as $key => $name ) {
						$headers[] = array(
							'label' => $name,
							'id'    => $key,
						);
					}
					$cached_headers[ $spreadsheet_id ][ $sheet_id ] = $headers;
					set_transient( 'gf_spreadsheet_headers', $cached_headers, 60 * 60 * 60 );
				} elseif ( isset( $response->error->errors ) ) {
					echo '<pre>';
					print_r( $response->error->errors );
					die;
				}
			}
		} else {
			$headers = $cached_headers[ $spreadsheet_id ][ $sheet_id ];
		}

		return $headers;
	}

	/**
	 * Add new row to GoogleSpreadsheet
	 *
	 * @since  1.0
	 * @author Aman Saini
	 * @return Status Code
	 */
	function add_new_row( $spreadsheet_id, $sheet_id, $row ) {
		$settings = get_option( 'gravityformsaddon_gfspreadsheet_settings' );
		$response = $this->make_request( 'sheets', 'POST', '/' . $spreadsheet_id . '/values/' . urlencode( $sheet_id ) . ':append?valueInputOption=USER_ENTERED', $row );

		if ( ! empty( $settings['debug_mode'] ) ) {
			echo '<pre>';
			echo '<strong>Debug Mode enabled::Please disable it after testing from Forms => Settings => GoogleSpreadsheet</strong><br/>';
			print_r( $response );
			die;
		}
	}

	public function make_request( $callscope, $method = 'GET', $uri = '', $params = null ) {
		$access_token = GFSpreadheet_Auth::get_access_token();
		if ( $callscope == 'drive' ) {
			// setup the request
			$url = 'https://www.googleapis.com' . $uri;
		} else {
			$url = 'https://sheets.googleapis.com/v4/spreadsheets' . $uri;
		}

		$headers = array(
			'content-type'  => 'application/json',
			'Authorization' => 'Bearer ' . $access_token,
		);

		// switch based on METHOD
		/***********
		// GET is for getting reecords
		// POST is for updating existing records and requires an ID
		*/

		// make the request
		$req_args = array(
			'method'    => $method,
			'headers'   => $headers,
			'sslverify' => false,  // set to true in live envrio
		);

		switch ( $method ) {
			case 'GET':
				if ( ! empty( $params ) ) {
					$querystring = urldecode( http_build_query( $params ) );
					// append a query string to the url
					$url = $url . '?' . $querystring;
				}

				break;
			case 'POST':
				if ( ! empty( $params ) ) {
					$req_args['body'] = json_encode( array( 'values' => $params ) );
				}
				break;

		}

		// make the remote request
		$result = wp_remote_request( $url, $req_args );

		// handle response
		if ( ! is_wp_error( $result ) ) {
			// no error
			$response = json_decode( wp_remote_retrieve_body( $result ) );
			return $response;

		} else {
			// error
			return $result->get_error_message();
		}

	}
}

function GF_GoogleSpreadsheet_Api() {
	return GF_GoogleSpreadsheet_Api::instance();
}
GF_GoogleSpreadsheet_Api();
