<?php
GFForms::include_feed_addon_framework();
class GFSpreadsheetFeedAddOn extends GFFeedAddOn {
	protected $_version                  = GFSPREADSHEET_VERSION;
	protected $_min_gravityforms_version = '1.9.11';
	protected $_slug                     = 'gfspreadsheet';
	protected $_path                     = 'gravity-forms-spreadsheet-addon/gravity-forms-spreadsheet-addon.php';
	protected $_full_path                = __FILE__;
	protected $_title                    = 'Gravity Forms Spreadsheet Add-On';
	protected $_short_title              = 'Google Spreadsheet Add-On';
	private static $_instance            = null;


	// Members plugin integration
	protected $_capabilities = array( 'gravityforms_spreadsheet', 'gravityforms_spreadsheet_uninstall' );

	// Permissions
	protected $_capabilities_settings_page = 'gravityforms_spreadsheet';
	protected $_capabilities_form_settings = 'gravityforms_spreadsheet';
	protected $_capabilities_uninstall     = 'gravityforms_spreadsheet_uninstall';

	// Sync
	protected $_sync_form_object = null;
	protected $_sync_feed_object = null;

	/**
	 * Get an instance of this class.
	 *
	 * @return GFSimpleFeedAddOn
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new GFSpreadsheetFeedAddOn();
		}
		return self::$_instance;
	}


	/**
	 * Plugin starting point. Handles hooks, loading of language files and PayPal delayed payment support.
	 */
	public function init() {
		parent::init();
		$this->add_delayed_payment_support(
			array(
				'option_label' => esc_html__( 'Add data to Google Spreadsheet only when payment is received.', 'gf-spreadsheet' ),
			)
		);
		add_action( 'before_loading_gf_spreasheet_feed', array( $this, 'maybe_clear_spreadsheet_cache' ) );

		add_action( 'wp_ajax_gf_spreadsheet_finish_code_exchange', 'GFSpreadheet_Auth::gf_finish_code_exchange' );
		add_action( 'wp_ajax_gf_spreadsheet_revoke', 'GFSpreadheet_Auth::revoke' );

		// sync entries
		add_action( 'wp_ajax_gf_sync_entries_google_spreadsheet', array( $this, 'sync_entries' ) );

	}


	// # SCRIPTS & STYLES -----------------------------------------------------------------------------------------------
	/**
	 * Return the scripts which should be enqueued.
	 *
	 * @return array
	 */
	public function scripts() {
		$scripts = array(
			array(
				'handle'  => 'gf_spreadsheet',
				'src'     => $this->get_base_url() . '/js/admin.js',
				'version' => $this->_version,
				'deps'    => array( 'jquery' ),
				'enqueue' => array(
					array(
						'admin_page' => array( 'form_settings' ),
						'tab'        => 'gfspreadsheet',
					),
				),
			),
		);
		return array_merge( parent::scripts(), $scripts );
	}

	// # FEED PROCESSING -----------------------------------------------------------------------------------------------
	/**
	 * Process the feed e.g. subscribe the user to a list.
	 *
	 * @param array   $feed  The feed object to be processed.
	 * @param array   $entry The entry object currently being processed.
	 * @param array   $form  The form object currently being processed.
	 * @return bool|void
	 */
	public function process_feed( $feed, $entry, $form ) {
		$headers   = array();
		$field_map = $this->get_field_map_fields( $feed, 'mappedFields' );

		foreach ( $field_map as $name => $field_id ) {
			if ( empty( $field_id ) ) {
				$headers[ $name ] ='';
				continue;
			}

			// send number field in correct format
			$field = RGFormsModel::get_field( $form, $field_id );

			if ( ! empty( $field ) && $field->type == 'number' ) {
				$value         = RGFormsModel::get_lead_field_value( $entry, $field );
				$headers[ $name ] = GFCommon::get_lead_field_display( $field, $value, $entry['currency'] );
			} elseif( $field_id == 'date_created' ) {
				// send entry date in correct timezone
				$headers[ $name ] =  GFCommon::format_date( $entry['date_created'], false, '' );
			} else {
				$value            = $this->get_field_value( $form, $entry, $field_id );
				$headers[ $name ] = $value;
			}
		}

		$spreadsheet_id = $feed['meta']['spreadsheet_id'];
		$sheet_id       = $feed['meta']['sheet_id'];

		if ( ! empty( $spreadsheet_id ) && ! empty( $sheet_id ) ) {
			$row            = array();
			$defaultheaders = GF_GoogleSpreadsheet_Api()->get_googlespreadsheet_headers( $spreadsheet_id, $sheet_id );

			if ( ! empty( $defaultheaders ) ) {
				foreach ( $defaultheaders as $key => $name ) {
					$row[ $key ] = isset( $headers[ $key ] ) ? $headers[ $key ] : '';
				}
				$final_row = array( $row );
				GF_GoogleSpreadsheet_Api()->add_new_row( $spreadsheet_id, $sheet_id, $final_row );
			}
		}

	}

	// # ADMIN FUNCTIONS -----------------------------------------------------------------------------------------------
	/**
	 * Configures the settings which should be rendered on the add-on settings tab.
	 *
	 * @return array
	 */
	public function plugin_settings_fields() {
		$doc_url = 'https://webholics.org/knowledgebase/get-client-id-client-secret-google-authorization-gravity-forms/';
		return array(
			array(
				'title'  => esc_html__( 'Spreadsheet Add-on Settings', 'gf-spreadsheet' ),
				'fields' => array(
					array(
						'name'    => 'client_id',
						'tooltip' => esc_html__( 'Google App client id', 'gf-spreadsheet' ),
						'label'   => esc_html__( 'Client ID', 'gf-spreadsheet' ),
						'type'    => 'text',
						'class'   => 'medium',
						'onblur'  => 'gf_spreasheet_trim_value(this)',
						'after_input' => '<br/> <i style="font-size:12px">You have to ceate a new project in googles console.<a href="' . $doc_url . '" target="_blank" >Learn how to get Client ID and Secret </a></i> ',
					),
					array(
						'name'        => 'client_secret',
						'tooltip'     => esc_html__( 'Google App client secret', 'gf-spreadsheet' ),
						'label'       => esc_html__( 'Client Secret', 'gf-spreadsheet' ),
						'type'        => 'text',
						'class'       => 'medium',
						'onblur'      => 'gf_spreasheet_trim_value(this)',
					),
					array(
						'name'        => 'client_token',
						'tooltip'     => esc_html__( 'Client Token Code', 'gf-spreadsheet' ),
						'label'       => esc_html__( 'Client Token Code', 'gf-spreadsheet' ),
						'type'        => 'text',
						'class'       => 'medium',
						'onblur'      => 'gf_spreasheet_trim_value(this)',
						'after_input' => '<br/> <i style="font-size:12px">Paste the Client Token Code here & click Update Settings button below to Authorize.</i>',
					),
					array(
						'label' => '',
						'type'  => 'authorize_account',
						'name'  => 'authorize_account',
					),
					array(
						'name'        => 'debug_mode',
						'tooltip'     => esc_html__( 'Enable debug mode to see google api response on form submit.', 'gf-spreadsheet' ),
						'label'       => esc_html__( 'Debug mode', 'gf-spreadsheet' ),
						'type'        => 'checkbox',

						'choices'     => array(
							array(
								'label' => 'Enable',
								'name'  => 'debug_mode',
							),
						),
						'after_input' => '<br/><i>Enable debug mode to see google api response on form submit.</i>',
					),
				),
			),
		);
	}
	/**
	 * Configures the settings which should be rendered on the feed edit page in the Form Settings > Simple Feed Add-On area.
	 *
	 * @return array
	 */
	public function feed_settings_fields() {

		$settings = $this->get_plugin_settings();
		do_action( 'before_loading_gf_spreasheet_feed' );
		return array(
			array(
				'title'  => esc_html__( 'Spreadsheet Feed Settings', 'gf-spreadsheet' ),
				'fields' => array(
					array(
						'label' => esc_html__( 'Feed name', 'gf-spreadsheet' ),
						'type'  => 'text',
						'name'  => 'feedName',
						'class' => 'medium',
					),
					array(
						'name'     => 'spreadsheet_id',
						'label'    => esc_html__( 'Select Spreadsheet', 'gf-spreadsheet' ),
						'type'     => 'spreadsheet_list',
						'required' => true,
						'tooltip'  => sprintf(
							'<h6>%s</h6>%s',
							esc_html__( 'Spreadsheet', 'gf-spreadsheet' ),
							esc_html__( 'Select the Spreadsheet to which you would like to add your form data.', 'gf-spreadsheet' )
						),
					),
				),
			),
			array(
				'dependency' => 'spreadsheet_id',
				'fields'     => array(
					array(
						'name'     => 'sheet_id',
						'label'    => esc_html__( 'Select Sheet', 'gf-spreadsheet' ),
						'type'     => 'sheet_list',
						'required' => true,
					),
				),
			),
			array(
				'dependency' => 'sheet_id',
				'fields'     => array(
					array(
						'name'      => 'mappedFields',
						'label'     => esc_html__( 'Map Sheet Headers', 'gf-spreadsheet' ),
						'type'      => 'field_map',
						'field_map' => $this->headers_map(),
					),
					array(
						'name'      => 'sync_entries',
						'label'     => esc_html__( 'Sync Entries', 'gf-spreadsheet' ),
						'type'      => 'sync_entries',
					),
					array(
						'name'           => 'condition',
						'label'          => esc_html__( 'Condition', 'gf-spreadsheet' ),
						'type'           => 'feed_condition',
						'checkbox_label' => esc_html__( 'Enable Condition', 'gf-spreadsheet' ),
						'instructions'   => esc_html__( 'Process this feed if', 'gf-spreadsheet' ),
					),
				),
			),
		);
	}


	/**
	 * Define the markup for the mailchimp_list type field.
	 *
	 * @since  3.0
	 * @access public
	 *
	 * @param array   $field The field properties.
	 * @param bool    $echo  Should the setting markup be echoed. Defaults to true.
	 * @return string
	 */
	public function settings_spreadsheet_list( $field, $echo = true ) {
		// Initialize HTML string.
		$html = '';
		// Files
		$files = GF_GoogleSpreadsheet_Api()->get_googlespreadsheet_files();

		// Initialize select options.
		$options = array(
			array(
				'label' => esc_html__( 'Select Spreadsheet', 'gf-spreadsheet' ),
				'value' => '',
			),
		);
		foreach ( $files as $spreadsheet ) {
			// Add list to select options.
			$options[] = array(
				'label' => esc_html( $spreadsheet['label'] ),
				'value' => esc_attr( $spreadsheet['id'] ),
			);
		}
		// Add select field properties.
		$field['type']     = 'select';
		$field['choices']  = $options;
		$field['onchange'] = 'jQuery(this).parents("form").submit();';
		// Generate select field.
		$html  = $this->settings_select( $field, false );
		$html .= '&nbsp;&nbsp;<button class=" button-secondary gf-spreadsheet-clear-cache" >Clear Spreadsheet Cache</button>';
		if ( $echo ) {
			echo $html;
		}
		return $html;
	}

	public function settings_sheet_list( $field, $echo = true ) {
		// Initialize HTML string.
		$html = '';
		// Get current list ID.
		$spreadsheet_id = $this->get_setting( 'spreadsheet_id' );
		// Sheets in File
		$sheets = json_decode( GF_GoogleSpreadsheet_Api()->get_googlespreadsheet_sheets( $spreadsheet_id ) );
		// Initialize select options.
		$options = array(
			array(
				'label' => esc_html__( 'Select Sheet', 'gf-spreadsheet' ),
				'value' => '',
			),
		);
		foreach ( $sheets as $sheet ) {
			// Add list to select options.
			$options[] = array(
				'label' => esc_html( $sheet->label ),
				'value' => esc_attr( $sheet->id ),
			);
		}
		// Add select field properties.
		$field['type']     = 'select';
		$field['choices']  = $options;
		$field['onchange'] = 'jQuery(this).parents("form").submit();';
		// Generate select field.
		$html = $this->settings_select( $field, false );
		if ( $echo ) {
			echo $html;
		}
		return $html;
	}

	public function settings_sync_entries( $field, $echo = true ) {
		$form_id           = rgget( 'id' );
		$feed_id           = rgget( 'fid' );
		$html = '<a data-form-id="' . $form_id . '" data-feed-id="' . $feed_id . '" id="googlespreadsheet-sync-gf-entries" class="button button" href="javascript:void(0)">Sync Entries</a>';
		$html .= '<span style="float:none" class="spreadsheet_sync_spinner spinner"></span>';
		$html .= '<span class="spreadsheet_sync_result"></span>';
		$html .= '<p class="description"> Note: This will append all the existing form entries data to the sheet you selected above, remember to map the sheet headers with form fields before sync. </p>';
		if ( $echo ) {
			echo $html;
		}
		return  $html;
	}

	function headers_map() {

		$spreadsheet_id = $this->get_setting( 'spreadsheet_id' );
		$sheet_id       = $this->get_setting( 'sheet_id' );
		$headers        = GF_GoogleSpreadsheet_Api()->get_googlespreadsheet_headers( $spreadsheet_id, $sheet_id );
		$field_map      = array();
		if ( ! empty( $headers ) ) {

			foreach ( $headers as  $header ) {
				$field_map[ $header['id'] ] = array(
					'name'  => $header['id'],
					'label' => $header['label'],
				);
			}
		}
		return $field_map;
	}


	function settings_authorize_account( $field ) {
		$notice         = '';
		$authUrl        = '';
		$settings       = $this->get_plugin_settings();
		$field['type']  = 'button';
		$field['name']  = 'gf_authorize_google_account';
		$field['class'] = 'gf_authorize_google_account button-secondary ';

		if ( ! rgar( $field, 'value' ) ) {
			$field['value'] = esc_html__( 'Generate Token', 'gf-spreadsheet' );
		}

		$client_id     = rgar( $settings, 'client_id' );
		$client_secret = rgar( $settings, 'client_secret' );
		$client_token = rgar( $settings, 'client_token' );
		$authsettings = get_option( 'gf_spreadsheet_auth' );

		if ( ! empty( $client_id ) && ! empty( $client_secret ) && ! empty( $client_token ) && empty( $authsettings['access_token'] ) ) {
			// Try setting access token & refresh token using auth code (client token)
			GFSpreadheet_Auth::set_oauth2_token( $client_token, 'auth_code' );
			$authsettings = get_option( 'gf_spreadsheet_auth' );
		}

		// var_dump($authsettings); die;
		if ( ! empty( $client_id ) && empty( $authsettings['access_token'] ) ) {
			$authUrl         = GFSPREADSHEET_OAUTH2_AUTH_URL . '?response_type=code&prompt=consent&access_type=offline&client_id=' . $client_id . '&redirect_uri=urn:ietf:wg:oauth:2.0:oob&scope=https://www.googleapis.com/auth/spreadsheets https://www.googleapis.com/auth/drive';
			$field['class'] .= 'gf_spreadsheet_authorization';
			$notice            = '   <span style="color:red" class="fa fa-times-circle"> Not Authorized</span>';
		} elseif ( ! empty( $client_id ) && ! empty( $authsettings['access_token'] ) ) {
			$authUrl         = GFSPREADSHEET_OAUTH2_REVOKE_URI . '/services/oauth2/revoke';
			$field['value']  = 'Revoke Token';
			$field['class'] .= 'gf_spreadsheet_deauthorize';
			$notice            = '   <span style="color:green" class="fa fa-check-circle"> Authorized </span>';
		} else {
			$field['disabled'] = 'disabled';
			$notice            = '<br/><i style="color:red">Please save Client ID & Client Secret first to Authorize</i>';
		}

		$attributes = $this->get_field_attributes( $field );

		$html  = '<a target="_blank" data-authurl="' . $authUrl . '"
					type="' . esc_attr( $field['type'] ) . '"
					 ' . implode( ' ', $attributes ) . '>' . $field['value'] . '</a>';
		$html .= $notice;
		echo $html; ?>


		<script type="text/javascript">
		function gf_spreasheet_trim_value(ele){
				ele.value= jQuery.trim(ele.value);
			}
		(function($){
			$(function(){
				// Authorization
				$('.gf_spreadsheet_authorization').click(function(e){

						var authUrl = $(this).data('authurl');
						if(authUrl == ''){
							e.preventDefault();
							alert('Please add and save Client ID and Client Secret first.')
							return false;
						}else{
							$(this).attr('href', authUrl);
							return true;
						}
					 })

				$('.gf_spreadsheet_deauthorize').click(function(e){
						e.preventDefault();

						 var data = {
							action: 'gf_spreadsheet_revoke',
						  };
						  $.post("<?php echo admin_url( 'admin-ajax.php' ); ?>", data, function(response) {
							  alert('Tokens revoked successfully')
							  //console.log(response);
							  location.reload();
						  });
				})

			})

		})(jQuery)
		</script>

		<?php
	}

	public function maybe_clear_spreadsheet_cache() {
		if ( isset( $_GET['frmclrcache'] ) ) {
			delete_transient( 'gf_spreadsheet_files' );
			delete_transient( 'gf_spreadsheet_sheets' );
			delete_transient( 'gf_spreadsheet_headers' );
		}
	}

	// change field label in field mapping area
	public function field_map_title() {
		return esc_html__( 'Sheet Headers', 'gf-spreadsheet' );
	}

	/**
	 * Configures which columns should be displayed on the feed list page.
	 *
	 * @return array
	 */
	public function feed_list_columns() {
		return array(
			'feedName' => esc_html__( 'Name', 'gf-spreadsheet' ),

		);
	}


	function sync_entries() {
		if ( ! isset( $_POST['formid'] ) || ! isset( $_POST['actionid'] ) ) {
			$result = array(
				'response' => 'error',
				'error_detail' => 'Error: FormId or FeedId not set. Please save feed first',
			);

			$this->send_ajax_response( $result );
		}
		$form_id = (int) $_POST['formid'];
		$feed_id = (int) $_POST['actionid'];

		// Set Sync Form Object
		if ( ! empty( $this->_sync_form_object ) ) {
			$form = $this->_sync_form_object;
		} else {
			$form = GFAPI::get_form( $form_id );
			$this->_sync_form_object = $form;
		}
		// Set Sync Feed Object
		if ( ! empty( $this->_sync_feed_object ) ) {
			$feed = $this->_sync_feed_object;
		} else {
			$feed = $this->get_feed( $feed_id );
			$this->_sync_feed_object = $feed;
		}

		$meta = rgar( $feed, 'meta' );
		// Add search filter in query if Conditional Logic is set
		$search_criteria = array();
		if ( $meta['feed_condition_conditional_logic'] ) {
			$conditional_logic = rgars( $feed, 'meta/feed_condition_conditional_logic_object/conditionalLogic' );
			$search_criteria['field_filters']['mode'] = $conditional_logic['logicType'];
			foreach ( $conditional_logic['rules'] as $rule ) {
				$search_criteria['field_filters'][] = array( 'key' => $rule['fieldId'], 'operator' =>  $rule['operator'], 'value' => $rule['value'] );
			}
		}

		$total_count     = 0;
		$step = isset( $_POST['step'] ) ? (int) sanitize_text_field( wp_unslash( $_POST['step'] ) ) : 1;
		$per_page = apply_filters( 'gf_googlespreadsheet_sync_per_page_limit', 10 );
		$offset = ( $step - 1 ) * $per_page;
		$paging = array( 'offset' => $offset, 'page_size' => $per_page );
		$entries = GFAPI::get_entries( $form_id, $search_criteria, array(), $paging, $total_count );
		//$this->send_ajax_response( $entries );
		if ( empty( $entries ) && $step == 1 ) {
			$result = array(
				'response' => 'error',
				'error_detail' => 'Error: Entries not found',
			);

			$this->send_ajax_response( $result );
		} else if ( empty( $entries ) ) {
			$result = array(
				'response' => 'success',
				'processed' => 'Sync completed.',
				'step' => 'complete',
			);

			$this->send_ajax_response( $result );
		}

		$headers = array();
		$final_rows = array();
		foreach ( $meta as $setting_name => $value ) {
			if ( $setting_name == 'spreadsheet_id' ) {
				$spreadsheet_id = $value;
			} else if ( $setting_name == 'sheet_id' ) {
				$sheet_id = $value;
			} else if ( strpos( $setting_name, 'mappedFields_' ) !== FALSE ) {
				$header_id = substr( $setting_name, 13 );
				$headers[$header_id] = $value; // this contains mapped field id
			}
		}

		if ( ! empty( $spreadsheet_id ) && ! empty( $sheet_id ) && ! empty( $headers ) ) {
			foreach ( $entries as $entry ) {
				$row = array();
				foreach ( $headers as $headerid => $field_id ) {
					$value = $this->get_field_value( $form, $entry, $field_id );
					$row[$headerid] = $value;
				}
				$final_rows[] = $row;
			}
		}

		$response = GF_GoogleSpreadsheet_Api()->make_request( 'sheets', 'POST', '/' . $spreadsheet_id . '/values/' . urlencode( $sheet_id ) . ':append?valueInputOption=USER_ENTERED', $final_rows );

		if ( isset( $response->updates ) ) {
			$rows = $response->updates->updatedRows;

			// Total entries processed so far
			$processed_entries_count = $offset + $total_count;
			$result = array(
				// translators: %d is the number of entries processed so far
				'processed' => sprintf( esc_html__( '%d entries processed.', 'gf-spreadsheet' ), $processed_entries_count ),
				'response' => 'success',
			);

			// Check if this is the last set of result
			if ( $total_count < $per_page ) {
				$result['step'] = 'complete';
			} else {
				$result['step'] = $step + 1;
			}
		} else {
			if ( isset( $response->error ) ) {
				$result = array(
					'response' => 'error',
					'error_detail' => 'Error:' . $response->error->message,
				);
			}else {
				$result = array(
					'response' => 'error',
					'error_detail' => 'Some error occured',
				);
			}

		}
		$this->send_ajax_response( $result );
	}

	public function send_ajax_response( $result ) {
		echo json_encode( $result );
		wp_die();
	}


	/**
	 * Enable feed duplication.
	 *
	 * @access public
	 * @param int|array $feed_id The ID of the feed to be duplicated or the feed object when duplicating a form.
	 * @return bool
	 */
	public function can_duplicate_feed( $feed_id ) {

		return true;

	}

	/**
	 * Prevent feeds being listed or created if an api key isn't valid.
	 *
	 * @return bool
	 */
	public function can_create_feed() {
		// Get the plugin settings.
		$settings = $this->get_plugin_settings();
		// Access a specific setting e.g. an api key
		$key = rgar( $settings, 'client_id' );
		return true;
	}

}
